'*************************************************************
' 
' ADOBE SYSTEMS INCORPORATED 
' Copyright 2005-2010 Adobe Systems Incorporated 
' All Rights Reserved 

' NOTICE:  Adobe permits you to use, modify, and 
' distribute this file in accordance with the terms
' of the Adobe license agreement accompanying it.  
' If you have received this file from a source 
' other than Adobe, then your use, modification,
' or distribution of it requires the prior 
' written permission of Adobe. 
' 
'*************************************************************

' ContactSheet.vbs

' DESCRIPTION

' This sample iterates through all images in a folder and 
' creates a named thumb nail for each of these images. The 
' number of rows and columns are hard-coded as the InputBox 
' VBScript function cannot be called within Illustrator.
' 
'*************************************************************

' Main Code - execution begins here

' Create the Application object and get a set of image files to work with
Set appRef = CreateObject("Illustrator.Application")
Set fileSystemObject = CreateObject("Scripting.FileSystemObject")

' Creating a folder browser in VBScript can be a problem (relying on either Windows API calls
' or use of ActiveX controls which may not be present on a given system). Instead, use
' Illustrator's built-in JavaScript to display a file browser. DoJavaScript can return a value,
' in this example it's the platform specific full path to the chosen folder.

doJavaScript = "var imagesFolder = Folder.selectDialog(""Select images folder:""); if (imagesFolder) folderPath = imagesFolder.fsName;"
imagesFolderPath = appRef.DoJavaScript(doJavaScript)

If (fileSystemObject.FolderExists(imagesFolderPath)) Then

	doJavaScript = "var numRows = prompt(""Enter number of rows:"", ""4""); if (numRows) numRows = parseInt(numRows); else numRows = 0;"
	numRows = appRef.DoJavaScript(doJavaScript)

	doJavaScript = "var numCols = prompt(""Enter number of columns:"", ""4""); if (numCols) numCols = parseInt(numCols); else numCols = 0;"
	numCols = appRef.DoJavaScript(doJavaScript)
	
	If (numRows > 0 And numCols > 0) Then
		imagesFolderPathTemp = Replace( imagesFolderPath, "\", "\\")
		doJavaScript = "alert(""Path to the test files used: " & CStr(imagesFolderPathTemp) & "    Number of rows: " & numRows & "  Number of columns: " & numCols & "  "")"
		appRef.DoJavaScript doJavaScript

		' Call function DoIt to iterate through all images in the folder 
		' and create the contact sheet
		Call DoIt(imagesFolderPath, CStr(numRows), CStr(numCols), 24, 22)
	Else
		doJavaScript = "alert(""Script requires at least 1 row and 1 column!"");"
		appRef.DoJavaScript doJavaScript
	End If

End If
'*************************************************************

' This routine iterates through all images in the folder and creates a named
' thumb nail for each of these

Sub DoIt(sDirName, verticalCount, horizontalCount, horizontalSpacing, verticalSpacing)
          
        ' Get the test files from the folder sDirName
        Set fso = CreateObject("Scripting.FileSystemObject")
	Set fldr = fso.GetFolder(sDirName)
	Set fls = fldr.Files
        numImages = verticalCount * horizontalCount
        Dim numFiles
        numFiles = fls.count
        Dim fileNames(5)
        
	
	' Add a new document
	Set docRef = appRef.Documents.Add
        
        ' Determine the dimensions of the document
        docPageOrigin = docRef.PageOrigin
        docLeft = docPageOrigin(0)
  	docTop = docRef.Height - docPageOrigin(1)
        printableWidth = docRef.Width - docLeft * 2
        printableHeight = docRef.Height - docPageOrigin(1) * 2
        
        ' Calculate the size of the individual grid spaces that the images will be placed in
        ' Dim gridWidth, gridHeight As Double
       
        gridWidth = (printableWidth + horizontalSpacing) / horizontalCount
        gridHeight = (printableHeight + verticalSpacing) / verticalCount
        
        ' Calculate the size of the individual images based on the printable document area,
        ' and on the number of images to be placed across and down the page
       
        imageWidth = gridWidth - horizontalSpacing
        imageHeight = gridHeight - verticalSpacing
           
        ' Normalize the image size so we end up with a square image
        
        If imageWidth < imageHeight Then
        	imageSize = imageWidth
                xAdjustment = 0
                yAdjustment = (imageHeight - imageWidth) / 2
        Else
                imageSize = imageHeight
                xAdjustment = (imageWidth - imageHeight) / 2
                yAdjustment = 0
        End If
            
        ' Calculate the bounding box for the first image to be placed
        ' Dim imageLeft, imageRight, imageTop, imageBottom As Double
        imageLeft = docLeft + xAdjustment
        imageRight = imageLeft + imageSize
            
        imageTop = docTop - yAdjustment
        imageBottom = imageTop - imageSize
            
        ' Iterate over the images in the list, positioning then one at a time
        ' After they are positioned, place a border around then to make them stand out better
        
        i = 1
        num = 0
        Dim aFile
        For each aFile in fls
		' Ignore system files.
		If not aFile.attributes and 4 Then
			num = num + 1
            		fileNames(num) = CStr(aFile)
		End If
        Next
        For j = 1 to numImages
        	
                Call AddRasterItemToPage(docRef,  fileNames(i), imageLeft, imageTop, (imageSize))
                    
                Call MakeFramingRectangle(docRef, imageLeft, imageTop, imageRight, imageBottom)
                Set fileObject = fso.getFile(fileNames(i))
                
                Call MakeTextLabel(docRef, CStr(fileObject.Name), imageLeft + imageSize / 2, imageBottom - 2)
                
                ' Calculate a new image position for the next iteration of the loop
                If j Mod horizontalCount <> 0 Then
                    ' Not at the end of row yet, move to next position in the row
                    imageLeft = imageLeft + gridWidth
                    imageRight = imageRight + gridWidth
                Else
                    ' If at the end of a row, first check to see if we have run out of rows
                    If j / horizontalCount >= verticalCount Then
                        Exit For
                    End If
                    
                    imageLeft = docLeft + xAdjustment
                    imageRight = imageLeft + imageSize
                        
                    imageTop = imageTop - gridHeight
                    imageBottom = imageBottom - gridHeight
                End If
                if (i = num) then
                    i = 1
                else
                    i = i + 1
                end if
        Next
            
        Set docRef = Nothing
        Set appRef = Nothing
End Sub

'*************************************************************

' This routine makes the labels for the thumbnails

Sub MakeTextLabel(aDocument, theText, xCenter, yVertPos)
                          
        ' Create the new text label
        Set aTextItem = aDocument.TextFrames.Add
        aTextItem.Contents = theText
        
        ' Calculate the final position and move the text label there
        aTextItem.Position = Array(xCenter - (aTextItem.Width / 2), yVertPos)
        
        ' Set the color of the text to default Illustrator color:
        ' No stroke & blank fill
        Set blackCMYK = CreateObject("Illustrator.CMYKColor")
        blackCMYK.Black = 100
        blackCMYK.Cyan = 0
        blackCMYK.Magenta = 0
        blackCMYK.Yellow = 0
        
        Set aTextRange = aTextItem.TextRange
        Set noStroke = CreateObject("Illustrator.NoColor")
        aTextRange.CharacterAttributes.StrokeColor = noStroke
        aTextRange.CharacterAttributes.FillColor = blackCMYK

End Sub

'*************************************************************

' This routine adds all images in the folder as RasterItems 
' to the Document

Sub AddRasterItemToPage(aDocument, theFile, imageLeft, imageTop, imageSize)
        
        ' Create a new raster item and link it to the image file
        Set aPlacedItem = aDocument.PlacedItems.Add
   	aPlacedItem.File = theFile
        
        ' Get the raster item dimensions
        itemWidth = aPlacedItem.Width
        itemHeight = aPlacedItem.Height
        
        ' Calculate the scale factors so the raster item can be fitted to the grid
        X_ScaleFactor = (imageSize / itemWidth) * 100
        Y_ScaleFactor = (imageSize / itemHeight) * 100
        
        ' Determine which of the scale factors to use
        If X_ScaleFactor < Y_ScaleFactor Then
            scaleFactor = X_ScaleFactor
        Else
            scaleFactor = Y_ScaleFactor
        End If
        
        ' Actually resize the raster item
        Call aPlacedItem.Resize(scaleFactor, scaleFactor)
                
        ' Get the new raster item dimensions
        itemWidth = aPlacedItem.Width
        itemHeight = aPlacedItem.Height
        
        ' Move the raster item to the middle of its grid cell
        aPlacedItem.Position = Array(imageLeft + (imageSize - itemWidth) / 2, _
                                     imageTop - (imageSize - itemHeight) / 2)
        
        Exit Sub

End Sub

'*************************************************************

Sub MakeFramingRectangle(aDocument, left, top, right, bottom)
        
        ' Add but not show the rectangle yet
       
        Set myPath = aDocument.PathItems.Rectangle(top, left, right - left, top - bottom)
        myPath.Hidden = True
        
        ' Make a black color
        Set blackCMYK = CreateObject("Illustrator.CMYKColor")
        
        blackCMYK.Black = 100
        blackCMYK.Cyan = 0
        blackCMYK.Magenta = 0
        blackCMYK.Yellow = 0
        
        ' Set the Color to 100% black
        myPath.StrokeColor = blackCMYK
        
        ' Make sure it is not filled
        myPath.Filled = False
        
        ' Show the resulting path
        myPath.Hidden = False
End Sub

'*************************************************************

'' SIG '' Begin signature block
'' SIG '' MIIpyQYJKoZIhvcNAQcCoIIpujCCKbYCAQExDzANBglg
'' SIG '' hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
'' SIG '' BgEEAYI3AgEeMCQCAQEEEE7wKRaZJ7VNj+Ws4Q8X66sC
'' SIG '' AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
'' SIG '' LruIkVvEnHeGhCBgDxIzLIGZrBboAzgy1h2T+tvNq82g
'' SIG '' gg6WMIIGsDCCBJigAwIBAgIQCK1AsmDSnEyfXs2pvZOu
'' SIG '' 2TANBgkqhkiG9w0BAQwFADBiMQswCQYDVQQGEwJVUzEV
'' SIG '' MBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQLExB3
'' SIG '' d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdpQ2Vy
'' SIG '' dCBUcnVzdGVkIFJvb3QgRzQwHhcNMjEwNDI5MDAwMDAw
'' SIG '' WhcNMzYwNDI4MjM1OTU5WjBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMIICIjANBgkq
'' SIG '' hkiG9w0BAQEFAAOCAg8AMIICCgKCAgEA1bQvQtAorXi3
'' SIG '' XdU5WRuxiEL1M4zrPYGXcMW7xIUmMJ+kjmjYXPXrNCQH
'' SIG '' 4UtP03hD9BfXHtr50tVnGlJPDqFX/IiZwZHMgQM+TXAk
'' SIG '' ZLON4gh9NH1MgFcSa0OamfLFOx/y78tHWhOmTLMBICXz
'' SIG '' ENOLsvsI8IrgnQnAZaf6mIBJNYc9URnokCF4RS6hnyzh
'' SIG '' GMIazMXuk0lwQjKP+8bqHPNlaJGiTUyCEUhSaN4QvRRX
'' SIG '' XegYE2XFf7JPhSxIpFaENdb5LpyqABXRN/4aBpTCfMjq
'' SIG '' GzLmysL0p6MDDnSlrzm2q2AS4+jWufcx4dyt5Big2MEj
'' SIG '' R0ezoQ9uo6ttmAaDG7dqZy3SvUQakhCBj7A7CdfHmzJa
'' SIG '' wv9qYFSLScGT7eG0XOBv6yb5jNWy+TgQ5urOkfW+0/tv
'' SIG '' k2E0XLyTRSiDNipmKF+wc86LJiUGsoPUXPYVGUztYuBe
'' SIG '' M/Lo6OwKp7ADK5GyNnm+960IHnWmZcy740hQ83eRGv7b
'' SIG '' UKJGyGFYmPV8AhY8gyitOYbs1LcNU9D4R+Z1MI3sMJN2
'' SIG '' FKZbS110YU0/EpF23r9Yy3IQKUHw1cVtJnZoEUETWJrc
'' SIG '' JisB9IlNWdt4z4FKPkBHX8mBUHOFECMhWWCKZFTBzCEa
'' SIG '' 6DgZfGYczXg4RTCZT/9jT0y7qg0IU0F8WD1Hs/q27Iwy
'' SIG '' CQLMbDwMVhECAwEAAaOCAVkwggFVMBIGA1UdEwEB/wQI
'' SIG '' MAYBAf8CAQAwHQYDVR0OBBYEFGg34Ou2O/hfEYb7/mF7
'' SIG '' CIhl9E5CMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/n
'' SIG '' upiuHA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAK
'' SIG '' BggrBgEFBQcDAzB3BggrBgEFBQcBAQRrMGkwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBB
'' SIG '' BggrBgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQw
'' SIG '' QwYDVR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGln
'' SIG '' aWNlcnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5j
'' SIG '' cmwwHAYDVR0gBBUwEzAHBgVngQwBAzAIBgZngQwBBAEw
'' SIG '' DQYJKoZIhvcNAQEMBQADggIBADojRD2NCHbuj7w6mdNW
'' SIG '' 4AIapfhINPMstuZ0ZveUcrEAyq9sMCcTEp6QRJ9L/Z6j
'' SIG '' fCbVN7w6XUhtldU/SfQnuxaBRVD9nL22heB2fjdxyyL3
'' SIG '' WqqQz/WTauPrINHVUHmImoqKwba9oUgYftzYgBoRGRjN
'' SIG '' YZmBVvbJ43bnxOQbX0P4PpT/djk9ntSZz0rdKOtfJqGV
'' SIG '' WEjVGv7XJz/9kNF2ht0csGBc8w2o7uCJob054ThO2m67
'' SIG '' Np375SFTWsPK6Wrxoj7bQ7gzyE84FJKZ9d3OVG3ZXQIU
'' SIG '' H0AzfAPilbLCIXVzUstG2MQ0HKKlS43Nb3Y3LIU/Gs4m
'' SIG '' 6Ri+kAewQ3+ViCCCcPDMyu/9KTVcH4k4Vfc3iosJocsL
'' SIG '' 6TEa/y4ZXDlx4b6cpwoG1iZnt5LmTl/eeqxJzy6kdJKt
'' SIG '' 2zyknIYf48FWGysj/4+16oh7cGvmoLr9Oj9FpsToFpFS
'' SIG '' i0HASIRLlk2rREDjjfAVKM7t8RhWByovEMQMCGQ8M4+u
'' SIG '' KIw8y4+ICw2/O/TOHnuO77Xry7fwdxPm5yg/rBKupS8i
'' SIG '' bEH5glwVZsxsDsrFhsP2JjMMB0ug0wcCampAMEhLNKhR
'' SIG '' ILutG4UI4lkNbcoFUCvqShyepf2gpx8GdOfy1lKQ/a+F
'' SIG '' SCH5Vzu0nAPthkX0tGFuv2jiJmCG6sivqf6UHedjGzqG
'' SIG '' VnhOMIIH3jCCBcagAwIBAgIQD99ExMbwCe2qSx8hp1o/
'' SIG '' EDANBgkqhkiG9w0BAQsFADBpMQswCQYDVQQGEwJVUzEX
'' SIG '' MBUGA1UEChMORGlnaUNlcnQsIEluYy4xQTA/BgNVBAMT
'' SIG '' OERpZ2lDZXJ0IFRydXN0ZWQgRzQgQ29kZSBTaWduaW5n
'' SIG '' IFJTQTQwOTYgU0hBMzg0IDIwMjEgQ0ExMB4XDTIzMTEw
'' SIG '' MzAwMDAwMFoXDTI1MTEwNDIzNTk1OVowgeYxEzARBgsr
'' SIG '' BgEEAYI3PAIBAxMCVVMxGTAXBgsrBgEEAYI3PAIBAhMI
'' SIG '' RGVsYXdhcmUxHTAbBgNVBA8MFFByaXZhdGUgT3JnYW5p
'' SIG '' emF0aW9uMRAwDgYDVQQFEwcyNzQ4MTI5MQswCQYDVQQG
'' SIG '' EwJVUzELMAkGA1UECBMCY2ExETAPBgNVBAcTCFNhbiBK
'' SIG '' b3NlMRMwEQYDVQQKEwpBZG9iZSBJbmMuMSwwKgYDVQQL
'' SIG '' EyNJbGx1c3RyYXRvciwgSW5EZXNpZ24sIEluQ29weSwg
'' SIG '' TXVzZTETMBEGA1UEAxMKQWRvYmUgSW5jLjCCAiIwDQYJ
'' SIG '' KoZIhvcNAQEBBQADggIPADCCAgoCggIBAPCR5c/cgyg4
'' SIG '' MpJD46/+BWENPhE1RjN77VE2BWlIUdiO2nlWelKthEq9
'' SIG '' aK+1ofKIlDVc9mvxksuXBLESi7HBt3PCVBntNJkjBJd0
'' SIG '' EXeprmDK8HZrz8y/KSwEewbTMjneMRIRcJV2GMoAyUzK
'' SIG '' EyFJF7m6ByeoNBDDXMD4AQF5C2l3wfvRbREw+G+cyeya
'' SIG '' 564AHpFJ9sfqi9jAycnEZISxEf6N72GgGQr04cZt3JTb
'' SIG '' TktZGt6uWdZnseXDLGRHzFu4P4EqmVKvfMDliaKyWBzV
'' SIG '' lY+gmtE1vSxIK79WTySFrFY5j1vtbKyHybk6tv4TOmLt
'' SIG '' Lw4OP/ngVDiw/ZP+b7saGxPOEfuh5LOAr+llt+6V5NhT
'' SIG '' dkYPR6cbokawwX2DbGg7IeVuUPdf9bx/LznFV/8kPxCB
'' SIG '' J6qg0dqvCmQt/XbBXPJUvDoVJ9z+9hDlrAoLESqpHK8U
'' SIG '' 73UFGohBMJ/7itIyKngem7Ms98z6x0i63FJA8C06g+XE
'' SIG '' GJUmR28+c6ba+m/sc6xHI4zdL5a84UeRS9+u8UL8N4av
'' SIG '' fpSxHYTTxVjk1ZlwQ8dLARwEVWvTu4lRcEck3RPogx7o
'' SIG '' o8wUpSw9++m+Oi+K/z7RP/R67uCEtrkdM5N/xUoQR/t6
'' SIG '' 3P2ia9g8+LuYJ2BNBd1+qMcSoSlDfIFLFMeeqESbMv+j
'' SIG '' UsRQJn2lZvufAgMBAAGjggICMIIB/jAfBgNVHSMEGDAW
'' SIG '' gBRoN+Drtjv4XxGG+/5hewiIZfROQjAdBgNVHQ4EFgQU
'' SIG '' RPnrJ9yT26rMrd/Q4PfQSlgeaRcwPQYDVR0gBDYwNDAy
'' SIG '' BgVngQwBAzApMCcGCCsGAQUFBwIBFhtodHRwOi8vd3d3
'' SIG '' LmRpZ2ljZXJ0LmNvbS9DUFMwDgYDVR0PAQH/BAQDAgeA
'' SIG '' MBMGA1UdJQQMMAoGCCsGAQUFBwMDMIG1BgNVHR8Ega0w
'' SIG '' gaowU6BRoE+GTWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0LmNv
'' SIG '' bS9EaWdpQ2VydFRydXN0ZWRHNENvZGVTaWduaW5nUlNB
'' SIG '' NDA5NlNIQTM4NDIwMjFDQTEuY3JsMFOgUaBPhk1odHRw
'' SIG '' Oi8vY3JsNC5kaWdpY2VydC5jb20vRGlnaUNlcnRUcnVz
'' SIG '' dGVkRzRDb2RlU2lnbmluZ1JTQTQwOTZTSEEzODQyMDIx
'' SIG '' Q0ExLmNybDCBlAYIKwYBBQUHAQEEgYcwgYQwJAYIKwYB
'' SIG '' BQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBc
'' SIG '' BggrBgEFBQcwAoZQaHR0cDovL2NhY2VydHMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZEc0Q29kZVNpZ25p
'' SIG '' bmdSU0E0MDk2U0hBMzg0MjAyMUNBMS5jcnQwCQYDVR0T
'' SIG '' BAIwADANBgkqhkiG9w0BAQsFAAOCAgEAPIZ9C4IuKg85
'' SIG '' 0H8M66OLMVllIpANmEjijIG/D65qP6jfma+2WFps2bb+
'' SIG '' vKKIEtG9TFtCAlSrA1OV6Ru78paq0gqw/q3QAN0OleFR
'' SIG '' F1NvT/TLO0ulhjXTETB1BwSKZaJH001DH0+60TmbGfVh
'' SIG '' mwLrEj2Zfl4COFwivZ/2tRoknZl5tmiu4hpjDbOapWd9
'' SIG '' 5cqliSoMeu2i588557eikdiDk6bXBa5MxHjyXvbvaIWN
'' SIG '' Zjsymn0L0sYkF3BToDN0h0XiTTZjZPHdSV2GjX6C/+Nj
'' SIG '' HSZGRtBaIxdAL1mKEWJV9qkEKELPb098rO+zGdn1E5IH
'' SIG '' pS1zSRIYngwZEjZfyoCkg1F+EW4QPs2+ff2CsW5i3IEd
'' SIG '' 7vL19w9oyce3S4wejvnlTKt+EQWn3gSPxrbkp3BxdWRN
'' SIG '' 25MKo5bS5bJ21BiwqcIJHcTxWNgaQPh4sZboi9IimCSn
'' SIG '' UenPQhqdeOuz0NnmepMEYd7L/kqnAeUZ8bW1UbITAWQ/
'' SIG '' UT3phjCfRQNpEh4OBoUcH8hOOHZ0Lh86mxv9lDP3i7c/
'' SIG '' bQzZnrJr2dS34JXolwREbTX9Yu5JB5X3YP539/M1Ms5O
'' SIG '' FZou/MHQBrWMsVBsEXDTWqwbMTYw/zvIiHXcGUnpzInw
'' SIG '' 1jQXlLrmwVHW4ug+A4LFO7bWC4mnZoDNWp0xnlGmbP6I
'' SIG '' ccJmW5C7RJMxghqLMIIahwIBATB9MGkxCzAJBgNVBAYT
'' SIG '' AlVTMRcwFQYDVQQKEw5EaWdpQ2VydCwgSW5jLjFBMD8G
'' SIG '' A1UEAxM4RGlnaUNlcnQgVHJ1c3RlZCBHNCBDb2RlIFNp
'' SIG '' Z25pbmcgUlNBNDA5NiBTSEEzODQgMjAyMSBDQTECEA/f
'' SIG '' RMTG8AntqksfIadaPxAwDQYJYIZIAWUDBAIBBQCggaIw
'' SIG '' GQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYB
'' SIG '' BAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcN
'' SIG '' AQkEMSIEIOta3RUDgdoAfUiUxTtkmDCCgFfCr1rCf4+x
'' SIG '' iRJ8rhpfMDYGCisGAQQBgjcCAQwxKDAmoCSAIgBBAGQA
'' SIG '' bwBiAGUAIABJAGwAbAB1AHMAdAByAGEAdABvAHIwDQYJ
'' SIG '' KoZIhvcNAQEBBQAEggIAhq8v4YYr/ruH3Pnjeidkh53n
'' SIG '' WqRdy3ZfUoW8JTPvX6VDnA5gUaZgKaJIERAJgtbNC6Tj
'' SIG '' pj65snLJ2JlsJQ+Pb8ZXqnBLcQ0HVwKCeie2foOqOI2i
'' SIG '' 1N7zPXHXAtzC6zrkBdDlqXpXf9vYSPaXLdyEqFbu3T20
'' SIG '' fzFhLk9hG6fKdcFYc7w2fIDI1wGYsDL67sYHXDGdf0Bo
'' SIG '' OvD59IsMhWtjr3DSzmmfwnA+NTi6fXNkuiFxXdgVw4E1
'' SIG '' y+swdpAhsxho7KwkCP1oZGdj2TELFtQ7i2ytFobHqTQs
'' SIG '' yqSxI/C6w2JpQpsaef635VDnloEofAA0drVkvpXlxbyU
'' SIG '' 9bYpSbZv39tz7C0mFN44Lhq4pVuD34tn9zLeNULICErU
'' SIG '' wm0fTtkSd601z9JGhy+mXNi0ZDxy7uH8kfHU/7NKRmlf
'' SIG '' V10wAQH94fbFXmN8TU/KHyu6zlymrLxUW49EGpk+crMq
'' SIG '' 7Fpd+mFn7k9iNgFZJ53ZfpdLo4/aTszuiaZoXzkY3iPU
'' SIG '' pajhcAvYzniXPhB6JKMKhmUBh4GZy/QfGogfBcXxnZ3C
'' SIG '' zKRBNPqOkHjRYrmgvFoxFAzM5gHGGeeQCqK90viOVyBR
'' SIG '' pryRZjlj0s+ncnb7QFIUMgr6HLKlYyIm/Gdzd0JBoU3w
'' SIG '' +ve0GjbPiybhIH4l7AIvd2+UWrAIUYpzWgAUslHLf7+h
'' SIG '' ghc6MIIXNgYKKwYBBAGCNwMDATGCFyYwghciBgkqhkiG
'' SIG '' 9w0BBwKgghcTMIIXDwIBAzEPMA0GCWCGSAFlAwQCAQUA
'' SIG '' MHgGCyqGSIb3DQEJEAEEoGkEZzBlAgEBBglghkgBhv1s
'' SIG '' BwEwMTANBglghkgBZQMEAgEFAAQgmmsDF8EMhRY/Smcu
'' SIG '' 23l/LHxdqijarfqphRJ3caMjNq8CEQDSjTBQnI6rbpRk
'' SIG '' n9Swbp0wGA8yMDI1MDQxNzE0MzkzMlqgghMDMIIGvDCC
'' SIG '' BKSgAwIBAgIQC65mvFq6f5WHxvnpBOMzBDANBgkqhkiG
'' SIG '' 9w0BAQsFADBjMQswCQYDVQQGEwJVUzEXMBUGA1UEChMO
'' SIG '' RGlnaUNlcnQsIEluYy4xOzA5BgNVBAMTMkRpZ2lDZXJ0
'' SIG '' IFRydXN0ZWQgRzQgUlNBNDA5NiBTSEEyNTYgVGltZVN0
'' SIG '' YW1waW5nIENBMB4XDTI0MDkyNjAwMDAwMFoXDTM1MTEy
'' SIG '' NTIzNTk1OVowQjELMAkGA1UEBhMCVVMxETAPBgNVBAoT
'' SIG '' CERpZ2lDZXJ0MSAwHgYDVQQDExdEaWdpQ2VydCBUaW1l
'' SIG '' c3RhbXAgMjAyNDCCAiIwDQYJKoZIhvcNAQEBBQADggIP
'' SIG '' ADCCAgoCggIBAL5qc5/2lSGrljC6W23mWaO16P2RHxjE
'' SIG '' iDtqmeOlwf0KMCBDEr4IxHRGd7+L660x5XltSVhhK64z
'' SIG '' i9CeC9B6lUdXM0s71EOcRe8+CEJp+3R2O8oo76EO7o5t
'' SIG '' Luslxdr9Qq82aKcpA9O//X6QE+AcaU/byaCagLD/GLoU
'' SIG '' b35SfWHh43rOH3bpLEx7pZ7avVnpUVmPvkxT8c2a2yC0
'' SIG '' WMp8hMu60tZR0ChaV76Nhnj37DEYTX9ReNZ8hIOYe4jl
'' SIG '' 7/r419CvEYVIrH6sN00yx49boUuumF9i2T8UuKGn9966
'' SIG '' fR5X6kgXj3o5WHhHVO+NBikDO0mlUh902wS/Eeh8F/UF
'' SIG '' aRp1z5SnROHwSJ+QQRZ1fisD8UTVDSupWJNstVkiqLq+
'' SIG '' ISTdEjJKGjVfIcsgA4l9cbk8Smlzddh4EfvFrpVNnes4
'' SIG '' c16Jidj5XiPVdsn5n10jxmGpxoMc6iPkoaDhi6JjHd5i
'' SIG '' bfdp5uzIXp4P0wXkgNs+CO/CacBqU0R4k+8h6gYldp4F
'' SIG '' CMgrXdKWfM4N0u25OEAuEa3JyidxW48jwBqIJqImd93N
'' SIG '' Rxvd1aepSeNeREXAu2xUDEW8aqzFQDYmr9ZONuc2MhTM
'' SIG '' izchNULpUEoA6Vva7b1XCB+1rxvbKmLqfY/M/SdV6mwW
'' SIG '' TyeVy5Z/JkvMFpnQy5wR14GJcv6dQ4aEKOX5AgMBAAGj
'' SIG '' ggGLMIIBhzAOBgNVHQ8BAf8EBAMCB4AwDAYDVR0TAQH/
'' SIG '' BAIwADAWBgNVHSUBAf8EDDAKBggrBgEFBQcDCDAgBgNV
'' SIG '' HSAEGTAXMAgGBmeBDAEEAjALBglghkgBhv1sBwEwHwYD
'' SIG '' VR0jBBgwFoAUuhbZbU2FL3MpdpovdYxqII+eyG8wHQYD
'' SIG '' VR0OBBYEFJ9XLAN3DigVkGalY17uT5IfdqBbMFoGA1Ud
'' SIG '' HwRTMFEwT6BNoEuGSWh0dHA6Ly9jcmwzLmRpZ2ljZXJ0
'' SIG '' LmNvbS9EaWdpQ2VydFRydXN0ZWRHNFJTQTQwOTZTSEEy
'' SIG '' NTZUaW1lU3RhbXBpbmdDQS5jcmwwgZAGCCsGAQUFBwEB
'' SIG '' BIGDMIGAMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5k
'' SIG '' aWdpY2VydC5jb20wWAYIKwYBBQUHMAKGTGh0dHA6Ly9j
'' SIG '' YWNlcnRzLmRpZ2ljZXJ0LmNvbS9EaWdpQ2VydFRydXN0
'' SIG '' ZWRHNFJTQTQwOTZTSEEyNTZUaW1lU3RhbXBpbmdDQS5j
'' SIG '' cnQwDQYJKoZIhvcNAQELBQADggIBAD2tHh92mVvjOIQS
'' SIG '' R9lDkfYR25tOCB3RKE/P09x7gUsmXqt40ouRl3lj+8Qi
'' SIG '' oVYq3igpwrPvBmZdrlWBb0HvqT00nFSXgmUrDKNSQqGT
'' SIG '' dpjHsPy+LaalTW0qVjvUBhcHzBMutB6HzeledbDCzFzU
'' SIG '' y34VarPnvIWrqVogK0qM8gJhh/+qDEAIdO/KkYesLyTV
'' SIG '' OoJ4eTq7gj9UFAL1UruJKlTnCVaM2UeUUW/8z3fvjxhN
'' SIG '' 6hdT98Vr2FYlCS7Mbb4Hv5swO+aAXxWUm3WpByXtgVQx
'' SIG '' iBlTVYzqfLDbe9PpBKDBfk+rabTFDZXoUke7zPgtd7/f
'' SIG '' vWTlCs30VAGEsshJmLbJ6ZbQ/xll/HjO9JbNVekBv2Tg
'' SIG '' em+mLptR7yIrpaidRJXrI+UzB6vAlk/8a1u7cIqV0yef
'' SIG '' 4uaZFORNekUgQHTqddmsPCEIYQP7xGxZBIhdmm4bhYsV
'' SIG '' A6G2WgNFYagLDBzpmk9104WQzYuVNsxyoVLObhx3Ruga
'' SIG '' EGru+SojW4dHPoWrUhftNpFC5H7QEY7MhKRyrBe7ucyk
'' SIG '' W7eaCuWBsBb4HOKRFVDcrZgdwaSIqMDiCLg4D+TPVgKx
'' SIG '' 2EgEdeoHNHT9l3ZDBD+XgbF+23/zBjeCtxz+dL/9NWR6
'' SIG '' P2eZRi7zcEO1xwcdcqJsyz/JceENc2Sg8h3KeFUCS7tp
'' SIG '' Fk7CrDqkMIIGrjCCBJagAwIBAgIQBzY3tyRUfNhHrP0o
'' SIG '' ZipeWzANBgkqhkiG9w0BAQsFADBiMQswCQYDVQQGEwJV
'' SIG '' UzEVMBMGA1UEChMMRGlnaUNlcnQgSW5jMRkwFwYDVQQL
'' SIG '' ExB3d3cuZGlnaWNlcnQuY29tMSEwHwYDVQQDExhEaWdp
'' SIG '' Q2VydCBUcnVzdGVkIFJvb3QgRzQwHhcNMjIwMzIzMDAw
'' SIG '' MDAwWhcNMzcwMzIyMjM1OTU5WjBjMQswCQYDVQQGEwJV
'' SIG '' UzEXMBUGA1UEChMORGlnaUNlcnQsIEluYy4xOzA5BgNV
'' SIG '' BAMTMkRpZ2lDZXJ0IFRydXN0ZWQgRzQgUlNBNDA5NiBT
'' SIG '' SEEyNTYgVGltZVN0YW1waW5nIENBMIICIjANBgkqhkiG
'' SIG '' 9w0BAQEFAAOCAg8AMIICCgKCAgEAxoY1BkmzwT1ySVFV
'' SIG '' xyUDxPKRN6mXUaHW0oPRnkyibaCwzIP5WvYRoUQVQl+k
'' SIG '' iPNo+n3znIkLf50fng8zH1ATCyZzlm34V6gCff1DtITa
'' SIG '' EfFzsbPuK4CEiiIY3+vaPcQXf6sZKz5C3GeO6lE98NZW
'' SIG '' 1OcoLevTsbV15x8GZY2UKdPZ7Gnf2ZCHRgB720RBidx8
'' SIG '' ald68Dd5n12sy+iEZLRS8nZH92GDGd1ftFQLIWhuNyG7
'' SIG '' QKxfst5Kfc71ORJn7w6lY2zkpsUdzTYNXNXmG6jBZHRA
'' SIG '' p8ByxbpOH7G1WE15/tePc5OsLDnipUjW8LAxE6lXKZYn
'' SIG '' LvWHpo9OdhVVJnCYJn+gGkcgQ+NDY4B7dW4nJZCYOjgR
'' SIG '' s/b2nuY7W+yB3iIU2YIqx5K/oN7jPqJz+ucfWmyU8lKV
'' SIG '' EStYdEAoq3NDzt9KoRxrOMUp88qqlnNCaJ+2RrOdOqPV
'' SIG '' A+C/8KI8ykLcGEh/FDTP0kyr75s9/g64ZCr6dSgkQe1C
'' SIG '' vwWcZklSUPRR8zZJTYsg0ixXNXkrqPNFYLwjjVj33GHe
'' SIG '' k/45wPmyMKVM1+mYSlg+0wOI/rOP015LdhJRk8mMDDtb
'' SIG '' iiKowSYI+RQQEgN9XyO7ZONj4KbhPvbCdLI/Hgl27Ktd
'' SIG '' RnXiYKNYCQEoAA6EVO7O6V3IXjASvUaetdN2udIOa5kM
'' SIG '' 0jO0zbECAwEAAaOCAV0wggFZMBIGA1UdEwEB/wQIMAYB
'' SIG '' Af8CAQAwHQYDVR0OBBYEFLoW2W1NhS9zKXaaL3WMaiCP
'' SIG '' nshvMB8GA1UdIwQYMBaAFOzX44LScV1kTN8uZz/nupiu
'' SIG '' HA9PMA4GA1UdDwEB/wQEAwIBhjATBgNVHSUEDDAKBggr
'' SIG '' BgEFBQcDCDB3BggrBgEFBQcBAQRrMGkwJAYIKwYBBQUH
'' SIG '' MAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0LmNvbTBBBggr
'' SIG '' BgEFBQcwAoY1aHR0cDovL2NhY2VydHMuZGlnaWNlcnQu
'' SIG '' Y29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcnQwQwYD
'' SIG '' VR0fBDwwOjA4oDagNIYyaHR0cDovL2NybDMuZGlnaWNl
'' SIG '' cnQuY29tL0RpZ2lDZXJ0VHJ1c3RlZFJvb3RHNC5jcmww
'' SIG '' IAYDVR0gBBkwFzAIBgZngQwBBAIwCwYJYIZIAYb9bAcB
'' SIG '' MA0GCSqGSIb3DQEBCwUAA4ICAQB9WY7Ak7ZvmKlEIgF+
'' SIG '' ZtbYIULhsBguEE0TzzBTzr8Y+8dQXeJLKftwig2qKWn8
'' SIG '' acHPHQfpPmDI2AvlXFvXbYf6hCAlNDFnzbYSlm/EUExi
'' SIG '' HQwIgqgWvalWzxVzjQEiJc6VaT9Hd/tydBTX/6tPiix6
'' SIG '' q4XNQ1/tYLaqT5Fmniye4Iqs5f2MvGQmh2ySvZ180HAK
'' SIG '' fO+ovHVPulr3qRCyXen/KFSJ8NWKcXZl2szwcqMj+sAn
'' SIG '' gkSumScbqyQeJsG33irr9p6xeZmBo1aGqwpFyd/EjaDn
'' SIG '' mPv7pp1yr8THwcFqcdnGE4AJxLafzYeHJLtPo0m5d2aR
'' SIG '' 8XKc6UsCUqc3fpNTrDsdCEkPlM05et3/JWOZJyw9P2un
'' SIG '' 8WbDQc1PtkCbISFA0LcTJM3cHXg65J6t5TRxktcma+Q4
'' SIG '' c6umAU+9Pzt4rUyt+8SVe+0KXzM5h0F4ejjpnOHdI/0d
'' SIG '' KNPH+ejxmF/7K9h+8kaddSweJywm228Vex4Ziza4k9Tm
'' SIG '' 8heZWcpw8De/mADfIBZPJ/tgZxahZrrdVcA6KYawmKAr
'' SIG '' 7ZVBtzrVFZgxtGIJDwq9gdkT/r+k0fNX2bwE+oLeMt8E
'' SIG '' ifAAzV3C+dAjfwAL5HYCJtnwZXZCpimHCUcr5n8apIUP
'' SIG '' /JiW9lVUKx+A+sDyDivl1vupL0QVSucTDh3bNzgaoSv2
'' SIG '' 7dZ8/DCCBY0wggR1oAMCAQICEA6bGI750C3n79tQ4ghA
'' SIG '' GFowDQYJKoZIhvcNAQEMBQAwZTELMAkGA1UEBhMCVVMx
'' SIG '' FTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UECxMQ
'' SIG '' d3d3LmRpZ2ljZXJ0LmNvbTEkMCIGA1UEAxMbRGlnaUNl
'' SIG '' cnQgQXNzdXJlZCBJRCBSb290IENBMB4XDTIyMDgwMTAw
'' SIG '' MDAwMFoXDTMxMTEwOTIzNTk1OVowYjELMAkGA1UEBhMC
'' SIG '' VVMxFTATBgNVBAoTDERpZ2lDZXJ0IEluYzEZMBcGA1UE
'' SIG '' CxMQd3d3LmRpZ2ljZXJ0LmNvbTEhMB8GA1UEAxMYRGln
'' SIG '' aUNlcnQgVHJ1c3RlZCBSb290IEc0MIICIjANBgkqhkiG
'' SIG '' 9w0BAQEFAAOCAg8AMIICCgKCAgEAv+aQc2jeu+RdSjww
'' SIG '' IjBpM+zCpyUuySE98orYWcLhKac9WKt2ms2uexuEDcQw
'' SIG '' H/MbpDgW61bGl20dq7J58soR0uRf1gU8Ug9SH8aeFaV+
'' SIG '' vp+pVxZZVXKvaJNwwrK6dZlqczKU0RBEEC7fgvMHhOZ0
'' SIG '' O21x4i0MG+4g1ckgHWMpLc7sXk7Ik/ghYZs06wXGXuxb
'' SIG '' Grzryc/NrDRAX7F6Zu53yEioZldXn1RYjgwrt0+nMNlW
'' SIG '' 7sp7XeOtyU9e5TXnMcvak17cjo+A2raRmECQecN4x7ax
'' SIG '' xLVqGDgDEI3Y1DekLgV9iPWCPhCRcKtVgkEy19sEcypu
'' SIG '' kQF8IUzUvK4bA3VdeGbZOjFEmjNAvwjXWkmkwuapoGfd
'' SIG '' pCe8oU85tRFYF/ckXEaPZPfBaYh2mHY9WV1CdoeJl2l6
'' SIG '' SPDgohIbZpp0yt5LHucOY67m1O+SkjqePdwA5EUlibaa
'' SIG '' RBkrfsCUtNJhbesz2cXfSwQAzH0clcOP9yGyshG3u3/y
'' SIG '' 1YxwLEFgqrFjGESVGnZifvaAsPvoZKYz0YkH4b235kOk
'' SIG '' GLimdwHhD5QMIR2yVCkliWzlDlJRR3S+Jqy2QXXeeqxf
'' SIG '' jT/JvNNBERJb5RBQ6zHFynIWIgnffEx1P2PsIV/EIFFr
'' SIG '' b7GrhotPwtZFX50g/KEexcCPorF+CiaZ9eRpL5gdLfXZ
'' SIG '' qbId5RsCAwEAAaOCATowggE2MA8GA1UdEwEB/wQFMAMB
'' SIG '' Af8wHQYDVR0OBBYEFOzX44LScV1kTN8uZz/nupiuHA9P
'' SIG '' MB8GA1UdIwQYMBaAFEXroq/0ksuCMS1Ri6enIZ3zbcgP
'' SIG '' MA4GA1UdDwEB/wQEAwIBhjB5BggrBgEFBQcBAQRtMGsw
'' SIG '' JAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmRpZ2ljZXJ0
'' SIG '' LmNvbTBDBggrBgEFBQcwAoY3aHR0cDovL2NhY2VydHMu
'' SIG '' ZGlnaWNlcnQuY29tL0RpZ2lDZXJ0QXNzdXJlZElEUm9v
'' SIG '' dENBLmNydDBFBgNVHR8EPjA8MDqgOKA2hjRodHRwOi8v
'' SIG '' Y3JsMy5kaWdpY2VydC5jb20vRGlnaUNlcnRBc3N1cmVk
'' SIG '' SURSb290Q0EuY3JsMBEGA1UdIAQKMAgwBgYEVR0gADAN
'' SIG '' BgkqhkiG9w0BAQwFAAOCAQEAcKC/Q1xV5zhfoKN0Gz22
'' SIG '' Ftf3v1cHvZqsoYcs7IVeqRq7IviHGmlUIu2kiHdtvRoU
'' SIG '' 9BNKei8ttzjv9P+Aufih9/Jy3iS8UgPITtAq3votVs/5
'' SIG '' 9PesMHqai7Je1M/RQ0SbQyHrlnKhSLSZy51PpwYDE3cn
'' SIG '' RNTnf+hZqPC/Lwum6fI0POz3A8eHqNJMQBk1RmppVLC4
'' SIG '' oVaO7KTVPeix3P0c2PR3WlxUjG/voVA9/HYJaISfb8rb
'' SIG '' II01YBwCA8sgsKxYoA5AY8WYIsGyWfVVa88nq2x2zm8j
'' SIG '' LfR+cWojayL/ErhULSd+2DrZ8LaHlv1b0VysGMNNn3O3
'' SIG '' AamfV6peKOK5lDGCA3YwggNyAgEBMHcwYzELMAkGA1UE
'' SIG '' BhMCVVMxFzAVBgNVBAoTDkRpZ2lDZXJ0LCBJbmMuMTsw
'' SIG '' OQYDVQQDEzJEaWdpQ2VydCBUcnVzdGVkIEc0IFJTQTQw
'' SIG '' OTYgU0hBMjU2IFRpbWVTdGFtcGluZyBDQQIQC65mvFq6
'' SIG '' f5WHxvnpBOMzBDANBglghkgBZQMEAgEFAKCB0TAaBgkq
'' SIG '' hkiG9w0BCQMxDQYLKoZIhvcNAQkQAQQwHAYJKoZIhvcN
'' SIG '' AQkFMQ8XDTI1MDQxNzE0MzkzMlowKwYLKoZIhvcNAQkQ
'' SIG '' AgwxHDAaMBgwFgQU29OF7mLb0j575PZxSFCHJNWGW0Uw
'' SIG '' LwYJKoZIhvcNAQkEMSIEILlt5uAXJqeBtVmIOVwn9FFp
'' SIG '' kAphkfyzVfSBhI2NXEYZMDcGCyqGSIb3DQEJEAIvMSgw
'' SIG '' JjAkMCIEIHZ2n6jyYy8fQws6IzCu1lZ1/tdz2wXWZbkF
'' SIG '' k5hDj5rbMA0GCSqGSIb3DQEBAQUABIICALmUi+akfWXH
'' SIG '' pJv0YdKRFBu6GlPX25jrtRoLIh3NLbcInFiebN1hL1TA
'' SIG '' 3rGQKaQ8e/H7ypioWBa6DiZ2GgcElLEIJmb0dur/cIcZ
'' SIG '' OaeRrujP6eD/UVdw9W8Sj3w7GozFlhqPhEx13pV+qjnH
'' SIG '' i+3qRmb4pJwffgRK4E5Eydbm5x8Mfgz7N2THv0ivbY6D
'' SIG '' 2fuzRLd9pdaRymWew9ea1twe0xIbrQgueg0Oxn5bGjXA
'' SIG '' /MxOWImygTL2SII0TkyFSD5EY+wCfvystDaus4uGflgM
'' SIG '' A37RiH0hfAis9x5J3EkqUMaSChB7pHDPPnhEBBraM6sx
'' SIG '' ZEkm/Sgi2DS4JHoebkdSKChWHtwOAEUwY18/KBHZcsYC
'' SIG '' nck1jZNVLGEMN6hZVoK2eoT9nq3OZf40+lfcYKOGOiG+
'' SIG '' XD0mWMS09jbb/dHNbtCB/1EsMxUCLU1WIgp9cAcJM92W
'' SIG '' fwQ0V5Pj019MfATQfDTKl6/A3vmrzQLW6LdxraO7ijiv
'' SIG '' B5JrlYT15xmtCdapi1guuR1VJaIxmaq1+ul97tFLSJec
'' SIG '' O3087XfuA+g6iMqL04XtNYcGu4ACQvCl3P6UmTmH2dkP
'' SIG '' FByEaHI4ulU7AZrYi0PVkOizbNEF4fvy3bSFwMI8RYrM
'' SIG '' zFQ+dq5WNfBYIyD3FFA7PVQE/5w2NqtciA4AQ+ubNA6L
'' SIG '' VeXXyQ3rGgSI
'' SIG '' End signature block
